<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Auth;

use App\Http\Controllers\UserController;
use App\Http\Controllers\ConquistaController;
use App\Http\Controllers\MinutosController;
use App\Http\Controllers\JogoModuloController;

// ---------- ROTAS PÚBLICAS (login, registro, esqueceu a senha) ----------

Route::get('/', function () {
    if (Auth::check()) return redirect()->route('dashboard');
    return view('welcome');
})->name('welcome');

Route::get('/login', fn() => view('auth.login'))->name('login');
Route::get('/register', fn() => view('auth.register'))->name('register');
Route::get('/forgot-password', fn() => view('auth.forgot-password'))->name('password.request');

// ---------- ROTAS PROTEGIDAS (somente para usuários autenticados) ----------

Route::middleware('auth')->group(function () {
    // ---------- PÁGINAS INICIAIS ----------
    Route::get('/home', fn() => view('home'));

    // Listas (precisam existir para os redirecionos do middleware)
    Route::get('/modulos', fn() => view('modulos/modulos'))->name('modulos');
    Route::get('/diarias', fn() => view('diarias/diarias'))->name('diarias');
    Route::get('/minutos', function () {
        if (view()->exists('minuto.minutos')) return view('minuto.minutos');
        if (view()->exists('minuto.index'))   return view('minuto.index');
        abort(404);
    })->name('minutos');

    // ---------- MÓDULOS 1–5 (com bloqueio por data) ----------
    Route::get('/modulo{modulo}', function ($modulo) {
        return view("modulos.modulo{$modulo}");
    })->where('modulo', '[1-5]')
      ->middleware('release:modulo');

    // POST legado (questionários etc.) também protegido
    Route::post('/update-resposta/modulo/{modulo}', [UserController::class, 'updateRespostaModulo'])
        ->name('update.resposta.modulo')
        ->middleware('release:modulo');

    // ---------- DIÁRIAS (PERGUNTAS) 1–5 (24h) ----------
    Route::get('/pergunta{diaria}', function ($diaria) {
        return view("diarias.pergunta{$diaria}");
    })->where('diaria', '[1-5]')
      ->middleware('release:pergunta');

    Route::post('/update-resposta/diarias/{diaria}', [UserController::class, 'updateRespostaDiaria'])
        ->name('update.resposta.diaria')
        ->middleware('release:pergunta');

    // ---------- MINUTO 1–5 (com bloqueio por data) ----------
    // Importante: /minuto{minuto} (sem barra) vem antes das rotas com barra
    Route::get('/minuto{minuto}', function ($minuto) {
        $view = "minuto.minuto{$minuto}";
        if (view()->exists($view)) return view($view);
        abort(404);
    })->where('minuto', '[1-5]')
      ->middleware('release:minuto');

    // Slugs e paths de minuto (mantidos como estavam)
    Route::get('/minuto/{pagina}', function ($pagina) {
        $viewDirect = "minuto.$pagina";
        $viewUnderscore = "minuto." . str_replace('-', '_', $pagina);
        if (view()->exists($viewDirect))   return view($viewDirect);
        if (view()->exists($viewUnderscore)) return view($viewUnderscore);
        abort(404);
    })->where('pagina', '[A-Za-z0-9\-_]+');

    Route::get('/minuto/{path}', function ($path) {
        $path = str_replace('-', '_', $path);
        $view = 'minuto.' . str_replace('/', '.', $path);
        if (view()->exists($view)) return view($view);
        abort(404);
    })->where('path', '.*');

    // POST para avaliar minuto (apenas uma rota; duplicidade removida)
    Route::post('/minutos/{minuto}/avaliar', [MinutosController::class, 'avaliar'])
        ->name('minutos.avaliar')
        ->middleware('release:minuto');

    // ---------- OUTRAS PÁGINAS ----------
    Route::get('/atos', fn() => view('/atos'));
    Route::get('/entenda', fn() => view('/entenda'));
    Route::get('/comojogar', fn() => view('/comojogar'));
    Route::get('/conquistas', [ConquistaController::class, 'index'])->name('conquistas');
    Route::get('/usuario', fn() => view('/usuario'));

    // ---------- JOGOS (controller único) ----------
    Route::post('/modulos/{modulo}/jogo', [JogoModuloController::class, 'store'])
        ->name('modulos.jogo.store')
        ->middleware('release:modulo');

    // ---------- DASHBOARD ----------
    Route::middleware([
        'auth:sanctum',
        config('jetstream.auth_session'),
        'verified',
    ])->group(function () {
        Route::get('/dashboard', fn() => view('dashboard'))->name('dashboard');
    });
});
