<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Route;
use App\Models\Log; // Adicionar a importação do modelo Log

class UserController extends Controller
{
    public function updateRespostaModulo(Request $request, $modulo)
    {
        $user   = Auth::user();
        $modulo = (int) $modulo;

        // Campo dinâmico: usu_resp_mod1, usu_resp_mod2, ...
        $campo = 'usu_resp_mod' . $modulo;

        if (!Schema::hasColumn('users', $campo)) {
            return $this->redirectToModulo($modulo)->with('error', 'Módulo inválido.');
        }

        // ===== Fluxo 1: formulário antigo (certo/errado) =====
        if ($request->filled('resposta')) {
            $request->validate([
                'resposta' => 'in:1,2',
            ]);

            $resposta     = (int) $request->input('resposta');
            $user->$campo = $resposta; // grava 1 (certa) ou 2 (errada)
            $user->save();

            // Log: Registra a resposta do usuário
            Log::create([
                'usu_id' => $user->id,
                'log_ip' => request()->ip(),
                'log_acao' => "respondeu o módulo {$modulo} com a resposta {$resposta}",
                'usu_email' => $user->email,
            ]);

            $flashKey = $resposta === 1 ? 'success' : 'error';
            $flashMsg = $resposta === 1
                ? 'Parabéns! Você acertou a questão.'
                : 'Poxa, você errou. Tente novamente.';

            return $this->redirectToModulo($modulo)->with($flashKey, $flashMsg);
        }

        // ===== Fluxo 2: formulário do JOGO =====
        $campoFinalizacao = 'modulo' . $modulo;
        if ($request->has($campoFinalizacao)) {
            // Ao concluir o jogo, marque o módulo como finalizado (= 1)
            $pontos        = (int) $request->input($campoFinalizacao);
            $user->$campo  = 3; // conforme sua regra: terminou -> 1
            $user->save();

            // Log: Registra a conclusão do módulo
            Log::create([
                'usu_id' => $user->id,
                'log_ip' => request()->ip(),
                'log_acao' => "concluiu o módulo {$modulo}",
                'usu_email' => $user->email,
            ]);

            return $this->redirectToModulo($modulo)
                ->with('success', 'Módulo concluído! Pontos: ' . $pontos);
        }

        // Se chegou aqui, requisição não traz dados esperados
        return $this->redirectToModulo($modulo)->with('error', 'Requisição inválida.');
    }

    public function updateRespostaDiaria(Request $request, $diaria)
    {
        $request->validate([
            'resposta' => 'required|in:1,2',
        ]);

        $user = Auth::user();

        $campoResposta = 'usu_resp_dia' . intval($diaria);
        $campoData = $campoResposta . '_data';

        if (!Schema::hasColumn('users', $campoResposta) || !Schema::hasColumn('users', $campoData)) {
            return redirect()->route('diarias')->with('error', 'Campo de resposta inválido.');
        }

        $resposta = $request->input('resposta');
        $user->$campoResposta = $resposta;
        $user->$campoData = now();
        $user->save();

        // Log: Registra a resposta diária
        Log::create([
            'usu_id' => $user->id,
            'log_ip' => request()->ip(),
            'log_acao' => "respondeu o desafio diário {$diaria} com a resposta {$resposta}",
            'usu_email' => $user->email,
        ]);

        $mensagem = $resposta == 1
            ? ['success' => 'Parabéns! Você acertou a questão.']
            : ['error' => 'Poxa, você errou. Volte amanhã para tentar uma nova pergunta'];

        return redirect()->route('diarias')->with($mensagem);
    }

    // Helper para redirecionar ao módulo atual (ajuste o nome da rota conforme seu app)
    protected function redirectToModulo(int $modulo)
    {
        if (Route::has('modulos.show')) {
            return redirect()->route('modulos.show', ['modulo' => $modulo]);
        }
        if (Route::has('modulo.show')) {
            return redirect()->route('modulo.show', ['modulo' => $modulo]);
        }
        return redirect()->back();
    }
}
